<?php
declare(strict_types=1);

require __DIR__ . '/helpers.php';
require __DIR__ . '/database.php';
require __DIR__ . '/telegram_api.php';

$allowWebExecution = defined('BROADCAST_WORKER_ALLOW_WEB') && BROADCAST_WORKER_ALLOW_WEB === true;
$config = require __DIR__ . '/config.php';

if (!$allowWebExecution && php_sapi_name() !== 'cli') {
    echo "This script must be run from the command line." . PHP_EOL;
    exit(1);
}

$argv = $argv ?? [];

$logFile = $config['app']['log_file'];
bot_log($logFile, 'broadcast_worker_invoked', [
    'argv' => $argv ?? [],
    'php_binary' => PHP_BINARY,
    'cwd' => getcwd(),
]);

$workerErrorFile = __DIR__ . '/worker-error.log';
register_shutdown_function(static function () use ($logFile, $workerErrorFile): void {
    $error = error_get_last();
    if ($error === null) {
        return;
    }

    $fatalTypes = [E_ERROR, E_PARSE, E_CORE_ERROR, E_COMPILE_ERROR, E_USER_ERROR];
    if (!in_array($error['type'], $fatalTypes, true)) {
        return;
    }

    $context = [
        'message' => $error['message'] ?? '',
        'file' => $error['file'] ?? '',
        'line' => $error['line'] ?? 0,
        'type' => $error['type'],
    ];

    bot_log($logFile, 'broadcast_worker_fatal', $context);

    $time = (new DateTimeImmutable('now', new DateTimeZone('Asia/Tehran')))->format('Y-m-d H:i:s');
    $payload = sprintf("[%s] %s in %s:%d\n", $time, $context['message'], $context['file'], $context['line']);
    @file_put_contents($workerErrorFile, $payload, FILE_APPEND);
});

// -----------------------------------------------------------------------------
// Optional debug logger for the worker (writes to worker.log)
// Enable by setting APP_DEBUG=1 or passing --debug as an argument
// -----------------------------------------------------------------------------
$workerDebugEnabled = (getenv('APP_DEBUG') === '1') || (isset($argv) && in_array('--debug', $argv, true));
$workerLogFile = __DIR__ . '/worker.log';
$worker_debug = function (string $message, array $context = []) use ($workerDebugEnabled, $workerLogFile): void {
    if (!$workerDebugEnabled) {
        return;
    }
    try {
        if (!is_file($workerLogFile)) {
            @touch($workerLogFile);
        }
        $time = (new DateTimeImmutable('now', new DateTimeZone('Asia/Tehran')))->format('Y-m-d H:i:s');
        $json = $context ? ' ' . json_encode($context, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) : '';
        @file_put_contents($workerLogFile, sprintf("[%s] %s%s\n", $time, $message, $json), FILE_APPEND);
    } catch (Throwable $e) {
        // best-effort only
    }
};

try {
    $pdo = Database::pdo($config['database']);
} catch (Throwable $exception) {
    bot_log($logFile, 'broadcast_worker_db_failed', ['error' => $exception->getMessage()]);
    $worker_debug('db_connection_failed', ['error' => $exception->getMessage()]);
    exit(1);
}

// startup diagnostics
$worker_debug('worker_boot', [
    'argv' => $argv ?? [],
    'cwd' => getcwd(),
    'php' => PHP_VERSION,
    'sapi' => php_sapi_name(),
]);

$jobId = (isset($argv[1]) && $argv[1] !== '--debug') ? (int) $argv[1] : 0;

if ($jobId > 0) {
    $job = fetch_broadcast_job($pdo, $jobId);
} else {
    $job = fetch_next_broadcast_job($pdo);
}

if (!$job) {
    bot_log($logFile, 'broadcast_worker_no_job');
    $worker_debug('no_job_found');
    exit(0);
}

if ($job['status'] === 'pending') {
    mark_broadcast_job_running($pdo, (int) $job['id']);
    $job = fetch_broadcast_job($pdo, (int) $job['id']);
}

if ($job['status'] !== 'running') {
    bot_log($logFile, 'broadcast_worker_skip_job', [
        'job_id' => $job['id'],
        'status' => $job['status'],
    ]);
    $worker_debug('skip_job', ['job_id' => $job['id'], 'status' => $job['status']]);
    exit(0);
}

$jobLockName = sprintf('broadcast_job_%d', (int) $job['id']);
if (!acquire_broadcast_job_lock($pdo, $jobLockName)) {
    bot_log($logFile, 'broadcast_worker_lock_skip', [
        'job_id' => $job['id'],
        'lock' => $jobLockName,
    ]);
    $worker_debug('lock_skip', ['job_id' => $job['id'], 'lock' => $jobLockName]);
    exit(0);
}

try {
    bot_log($logFile, 'broadcast_worker_start', ['job_id' => $job['id']]);
    $worker_debug('job_running', ['job_id' => $job['id']]);

    // Ensure the source is accessible (copy to admin group if available)
    $job = ensure_broadcast_source_accessible($pdo, $config, $job, $worker_debug);

    refresh_progress_message($pdo, $config, $job, false);

    $token = $config['telegram']['bot_token'];

    while (true) {
        try {
            $job = fetch_broadcast_job($pdo, (int) $job['id']);
            if (!$job) {
                $worker_debug('job_missing_after_fetch');
                break;
            }

            $autoDeleteSeconds = (int) ($job['auto_delete_seconds'] ?? 0);
            $pinMessage = (int) ($job['pin_message'] ?? 0) === 1;

            process_due_broadcast_deletes($pdo, $config, (int) $job['id']);

            if ($job['status'] === 'cancelled') {
                refresh_progress_message($pdo, $config, $job, true);
                bot_log($logFile, 'broadcast_worker_cancelled', ['job_id' => $job['id']]);
                $worker_debug('job_cancelled', ['job_id' => $job['id']]);
                break;
            }

            $queue = fetch_broadcast_queue_batch($pdo, (int) $job['id']);
            $worker_debug('queue_batch', ['job_id' => (int) $job['id'], 'count' => is_array($queue) ? count($queue) : 0]);
            if ($queue === []) {
                mark_broadcast_job_completed($pdo, (int) $job['id']);
                $job = fetch_broadcast_job($pdo, (int) $job['id']);
                refresh_progress_message($pdo, $config, $job, true);
                notify_broadcast_finished($config, $job);
                bot_log($logFile, 'broadcast_worker_completed', ['job_id' => $job['id']]);
                if ($autoDeleteSeconds > 0) {
                    drain_broadcast_deletes($pdo, $config, $job);
                }
                $worker_debug('job_completed', ['job_id' => $job['id']]);
                break;
            }

            foreach ($queue as $item) {
                $queueId = (int) $item['queue_id'];
                $telegramId = (int) $item['telegram_id'];

                if ($telegramId === 0) {
                    mark_queue_failed($pdo, $queueId, 'telegram_id_missing');
                    increment_failed_count($pdo, (int) $job['id']);
                    $worker_debug('queue_invalid_telegram_id', ['queue_id' => $queueId]);
                    continue;
                }

                try {
                    $worker_debug('copyMessage', ['to' => $telegramId, 'from' => (int) $job['source_chat_id'], 'id' => (int) $job['source_message_id']]);
                    $result = telegram_request($token, 'copyMessage', [
                        'chat_id' => $telegramId,
                        'from_chat_id' => (int) $job['source_chat_id'],
                        'message_id' => (int) $job['source_message_id'],
                    ]);

                    $messageId = (int) ($result['message_id'] ?? 0);

                    if ($pinMessage && $messageId > 0) {
                        try {
                            telegram_request($token, 'pinChatMessage', [
                                'chat_id' => $telegramId,
                                'message_id' => $messageId,
                                'disable_notification' => true,
                            ]);
                            $worker_debug('pin_ok', ['chat_id' => $telegramId, 'message_id' => $messageId]);
                        } catch (Throwable $pinException) {
                            bot_log($logFile, 'broadcast_pin_failed', [
                                'job_id' => $job['id'],
                                'user_id' => $item['user_id'],
                                'error' => $pinException->getMessage(),
                            ]);
                            $worker_debug('pin_failed', ['chat_id' => $telegramId, 'error' => $pinException->getMessage()]);
                        }
                    }

                    if ($autoDeleteSeconds > 0 && $messageId > 0) {
                        schedule_broadcast_delete($pdo, (int) $job['id'], $telegramId, $messageId, $autoDeleteSeconds);
                        $worker_debug('delete_scheduled', ['chat_id' => $telegramId, 'message_id' => $messageId, 'after' => $autoDeleteSeconds]);
                    }

                    mark_queue_sent($pdo, $queueId);
                    increment_sent_count($pdo, (int) $job['id']);
                    $worker_debug('queue_sent', ['queue_id' => $queueId, 'chat_id' => $telegramId]);
                } catch (Throwable $exception) {
                    $error = $exception->getMessage();
                    mark_queue_failed($pdo, $queueId, $error);
                    increment_failed_count($pdo, (int) $job['id']);
                    bot_log($logFile, 'broadcast_send_failed', [
                        'job_id' => $job['id'],
                        'queue_id' => $queueId,
                        'user_id' => $item['user_id'],
                        'error' => $error,
                    ]);
                    $worker_debug('send_failed', ['queue_id' => $queueId, 'chat_id' => $telegramId, 'error' => $error]);

                    if (str_contains($error, 'blocked by the user') || str_contains($error, 'user is deactivated')) {
                        mark_user_blocked($pdo, (int) $item['user_id']);
                        $worker_debug('marked_user_blocked', ['user_id' => (int) $item['user_id']]);
                    }
                }
            }

            $job = fetch_broadcast_job($pdo, (int) $job['id']);
            refresh_progress_message($pdo, $config, $job, false);

            process_due_broadcast_deletes($pdo, $config, (int) $job['id']);

            sleep(6);
        } catch (Throwable $loopException) {
            bot_log($logFile, 'broadcast_worker_loop_exception', [
                'job_id' => $job['id'] ?? null,
                'error' => $loopException->getMessage(),
            ]);
            $worker_debug('loop_exception', ['error' => $loopException->getMessage()]);
            // avoid tight loop on fatal per-iteration issues
            sleep(2);
        }
    }

    bot_log($logFile, 'broadcast_worker_end', ['job_id' => $job['id'] ?? null]);

    drain_broadcast_deletes($pdo, $config, $job ?? null);
} finally {
    release_broadcast_job_lock($pdo, $jobLockName);
}

// -----------------------------------------------------------------------------
// Helper functions
// -----------------------------------------------------------------------------

/**
 * Update the progress message for admins with the latest counts.
 */
function refresh_progress_message(PDO $pdo, array $config, ?array $job, bool $finished): void
{
    if (!$job) {
        return;
    }

    $chatId = (int) ($job['progress_chat_id'] ?? 0);
    $messageId = (int) ($job['progress_message_id'] ?? 0);
    if ($chatId === 0 || $messageId === 0) {
        return;
    }

    $text = build_broadcast_progress_text($job);
    $keyboard = build_broadcast_progress_keyboard((int) ($job['id'] ?? 0), $finished);

    try {
        telegram_request($config['telegram']['bot_token'], 'editMessageText', [
            'chat_id' => $chatId,
            'message_id' => $messageId,
            'text' => $text,
            'reply_markup' => $keyboard,
            'parse_mode' => 'HTML',
            'disable_web_page_preview' => true,
        ]);
    } catch (Throwable $exception) {
        bot_log($config['app']['log_file'], 'broadcast_worker_progress_failed', [
            'job_id' => $job['id'] ?? null,
            'error' => $exception->getMessage(),
        ]);
    }
}

/**
 * Acquire a MySQL named lock to prevent concurrent workers from processing the same job.
 */
function acquire_broadcast_job_lock(PDO $pdo, string $lockName): bool
{
    try {
        $stmt = $pdo->prepare('SELECT GET_LOCK(:name, 0)');
        $stmt->execute([':name' => $lockName]);
        return (int) $stmt->fetchColumn() === 1;
    } catch (Throwable $exception) {
        return false;
    }
}

/**
 * Release a previously acquired MySQL named lock.
 */
function release_broadcast_job_lock(PDO $pdo, string $lockName): void
{
    try {
        $stmt = $pdo->prepare('SELECT RELEASE_LOCK(:name)');
        $stmt->execute([':name' => $lockName]);
    } catch (Throwable $exception) {
        // Ignore release failures.
    }
}

/**
 * Notify the job creator that the broadcast finished.
 */
function notify_broadcast_finished(array $config, ?array $job): void
{
    if (!$job) {
        return;
    }

    $chatId = (int) ($job['progress_chat_id'] ?? $job['created_by'] ?? 0);
    if ($chatId === 0) {
        return;
    }

    $sent = (int) ($job['sent_count'] ?? 0);
    $failed = (int) ($job['failed_count'] ?? 0);
    $total = (int) ($job['total_count'] ?? 0);
    $pending = max(0, $total - $sent - $failed);

    $message = sprintf(
        "ارسال همگانی با موفقیت تمام شد 🎉\n✅ ارسال موفق: %s کاربر\n❌ ارسال ناموفق: %s کاربر\n⏳ در انتظار: %s کاربر",
        number_format($sent),
        number_format($failed),
        number_format($pending)
    );

    try {
        telegram_send_message($config['telegram']['bot_token'], $chatId, $message, [
            'reply_markup' => make_inline_keyboard([
                [[
                    'text' => 'بازگشت ↩️',
                    'callback_data' => 'admin_menu',
                ]],
            ]),
        ]);
    } catch (Throwable $exception) {
        bot_log($config['app']['log_file'], 'broadcast_notify_failed', [
            'job_id' => $job['id'] ?? null,
            'error' => $exception->getMessage(),
        ]);
    }
}

/**
 * Ensure the job's source message is in a chat the bot fully controls.
 * If an admin_group_id is configured, we copy the source message into that group
 * and rewrite job.source_chat_id/message_id so all subsequent copies are from there.
 */
function ensure_broadcast_source_accessible(PDO $pdo, array $config, array $job, callable $debug): array
{
    $groupId = (int) ($config['telegram']['admin_group_id'] ?? 0);
    if ($groupId === 0) {
        return $job;
    }

    // If the job already points at the group, nothing to do
    if ((int) ($job['source_chat_id'] ?? 0) === $groupId) {
        return $job;
    }

    $token = $config['telegram']['bot_token'];
    $sourceChat = (int) ($job['source_chat_id'] ?? 0);
    $sourceMsg  = (int) ($job['source_message_id'] ?? 0);
    if ($sourceChat === 0 || $sourceMsg === 0) {
        return $job;
    }

    try {
        $debug('cache_source_copy_try', ['from_chat' => $sourceChat, 'msg' => $sourceMsg, 'to_group' => $groupId]);
        $result = telegram_request($token, 'copyMessage', [
            'chat_id' => $groupId,
            'from_chat_id' => $sourceChat,
            'message_id' => $sourceMsg,
        ]);
        $newMsgId = (int) ($result['message_id'] ?? 0);
        if ($newMsgId > 0) {
            // rewrite job to use group as source
            $stmt = $pdo->prepare('UPDATE broadcast_jobs SET source_chat_id = :chat, source_message_id = :mid WHERE id = :id');
            $stmt->execute([':chat' => $groupId, ':mid' => $newMsgId, ':id' => (int) $job['id']]);
            $job['source_chat_id'] = $groupId;
            $job['source_message_id'] = $newMsgId;
            $debug('cache_source_copy_ok', ['new_message_id' => $newMsgId]);
        }
    } catch (Throwable $e) {
        // If this fails we still try to use original source; errors will be logged per-recipient
        $debug('cache_source_copy_failed', ['error' => $e->getMessage()]);
    }

    return $job;
}

function fetch_broadcast_job(PDO $pdo, int $jobId): ?array
{
    $stmt = $pdo->prepare('SELECT * FROM broadcast_jobs WHERE id = :id LIMIT 1');
    $stmt->execute([':id' => $jobId]);
    $row = $stmt->fetch();
    return $row ?: null;
}

function fetch_next_broadcast_job(PDO $pdo): ?array
{
    $stmt = $pdo->query("SELECT * FROM broadcast_jobs WHERE status IN ('pending','running') ORDER BY FIELD(status,'running','pending'), created_at ASC LIMIT 1");
    $row = $stmt->fetch();
    return $row ?: null;
}

function mark_broadcast_job_running(PDO $pdo, int $jobId): void
{
    $stmt = $pdo->prepare("UPDATE broadcast_jobs SET status = 'running', started_at = NOW() WHERE id = :id AND status = 'pending'");
    $stmt->execute([':id' => $jobId]);
}

function mark_broadcast_job_completed(PDO $pdo, int $jobId): void
{
    $stmt = $pdo->prepare("UPDATE broadcast_jobs SET status = 'completed', finished_at = NOW() WHERE id = :id");
    $stmt->execute([':id' => $jobId]);
}

function fetch_broadcast_queue_batch(PDO $pdo, int $jobId): array
{
    // Only real user chats (telegram_id > 0), skip channels/supergroups
    $stmt = $pdo->prepare("
        SELECT q.id AS queue_id, q.user_id, u.telegram_id
        FROM broadcast_queue q
        INNER JOIN bot_users u ON u.id = q.user_id
        WHERE q.job_id = :job_id
          AND q.status = 'pending'
          AND u.telegram_id > 0
        ORDER BY q.id ASC
        LIMIT 30
    ");
    $stmt->execute([':job_id' => $jobId]);
    return $stmt->fetchAll() ?: [];
}

function mark_queue_sent(PDO $pdo, int $queueId): void
{
    $stmt = $pdo->prepare("UPDATE broadcast_queue SET status = 'sent', last_attempt_at = NOW(), last_error = NULL WHERE id = :id");
    $stmt->execute([':id' => $queueId]);
}

function mark_queue_failed(PDO $pdo, int $queueId, string $error): void
{
    $stmt = $pdo->prepare("UPDATE broadcast_queue SET status = 'failed', last_attempt_at = NOW(), last_error = :error WHERE id = :id");
    $stmt->execute([
        ':id' => $queueId,
        ':error' => mb_substr($error, 0, 250),
    ]);
}

function increment_sent_count(PDO $pdo, int $jobId): void
{
    $pdo->prepare('UPDATE broadcast_jobs SET sent_count = sent_count + 1 WHERE id = :id')->execute([':id' => $jobId]);
}

function increment_failed_count(PDO $pdo, int $jobId): void
{
    $pdo->prepare('UPDATE broadcast_jobs SET failed_count = failed_count + 1 WHERE id = :id')->execute([':id' => $jobId]);
}

function mark_user_blocked(PDO $pdo, int $userId): void
{
    $stmt = $pdo->prepare('UPDATE bot_users SET has_blocked_bot = 1 WHERE id = :id');
    $stmt->execute([':id' => $userId]);
}

function schedule_broadcast_delete(PDO $pdo, int $jobId, int $chatId, int $messageId, int $ttlSeconds): void
{
    if ($ttlSeconds <= 0) {
        return;
    }

    $deleteAt = (new DateTimeImmutable('now', new DateTimeZone('Asia/Tehran')))->modify(sprintf('+%d seconds', $ttlSeconds));

    $stmt = $pdo->prepare('INSERT INTO broadcast_deletes (job_id, chat_id, message_id, delete_at) VALUES (:job_id, :chat_id, :message_id, :delete_at)');
    $stmt->execute([
        ':job_id' => $jobId,
        ':chat_id' => $chatId,
        ':message_id' => $messageId,
        ':delete_at' => $deleteAt->format('Y-m-d H:i:s'),
    ]);
}

function process_due_broadcast_deletes(PDO $pdo, array $config, int $jobId, int $limit = 100): int
{
    $stmt = $pdo->prepare('SELECT id, chat_id, message_id FROM broadcast_deletes WHERE job_id = :job_id AND delete_at <= NOW() ORDER BY id ASC LIMIT :limit');
    $stmt->bindValue(':job_id', $jobId, PDO::PARAM_INT);
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->execute();

    $rows = $stmt->fetchAll();
    if (!$rows) {
        return 0;
    }

    $token = $config['telegram']['bot_token'];
    $logFile = $config['app']['log_file'];

    foreach ($rows as $row) {
        try {
            telegram_request($token, 'deleteMessage', [
                'chat_id' => (int) $row['chat_id'],
                'message_id' => (int) $row['message_id'],
            ]);
        } catch (Throwable $exception) {
            bot_log($logFile, 'broadcast_delete_failed', [
                'job_id' => $jobId,
                'chat_id' => $row['chat_id'],
                'message_id' => $row['message_id'],
                'error' => $exception->getMessage(),
            ]);
        }

        $deleteStmt = $pdo->prepare('DELETE FROM broadcast_deletes WHERE id = :id');
        $deleteStmt->execute([':id' => $row['id']]);
    }

    return count($rows);
}

function pending_broadcast_deletes_count(PDO $pdo, int $jobId): int
{
    $stmt = $pdo->prepare('SELECT COUNT(*) FROM broadcast_deletes WHERE job_id = :job_id');
    $stmt->execute([':job_id' => $jobId]);
    return (int) $stmt->fetchColumn();
}

function drain_broadcast_deletes(PDO $pdo, array $config, ?array $job): void
{
    if (!$job) {
        return;
    }

    $jobId = (int) ($job['id'] ?? 0);
    if ($jobId === 0) {
        return;
    }

    $ttl = (int) ($job['auto_delete_seconds'] ?? 0);
    if ($ttl <= 0) {
        return;
    }

    $deadline = time() + $ttl + 5;
    while (time() <= $deadline) {
        $processed = process_due_broadcast_deletes($pdo, $config, $jobId);
        $remaining = pending_broadcast_deletes_count($pdo, $jobId);
        if ($remaining === 0) {
            break;
        }
        if ($processed === 0) {
            sleep(5);
        }
    }
}
